document.querySelectorAll('.player').forEach(p => p.addEventListener('contextmenu', e => e.preventDefault()));

(function idleRedirect() {
  const IDLE_TIME = 30 * 60 * 1000; // 30 Minuten in ms
  let idleTimer;

  function resetTimer() {
    clearTimeout(idleTimer);
    idleTimer = setTimeout(() => {
      showPage('start');
    }, IDLE_TIME);
  }

  const events = [
    'mousemove',
    'mousedown',
    'keydown',
    'scroll',
    'touchstart'
  ];

  events.forEach(event =>
    window.addEventListener(event, resetTimer, true)
  );

  resetTimer();
})();

const initModal = (modal) => {
    if (!modal) return;
    if (modal.dataset.inited) return;

    const video = modal.querySelector('video');
    const toggle = modal.querySelector('.btn');
    const icon = modal.querySelector('.icon');
    const range = modal.querySelector('input[type="range"]');

    if (!video || !toggle || !icon || !range) {
        modal.dataset.inited = '1';
        return;
    }

    const setIcon = () => {
        const playing = !video.paused && !video.ended;
        icon.className = 'icon ' + (playing ? 'pause' : 'play');
    };

    const setRangeVisual = (val) => {
        const pct = Math.max(0, Math.min(100, val));
        range.style.backgroundSize = pct + '% 100%';
    };

    const syncUI = () => {
        const dur = video.duration || 0;
        const cur = video.currentTime || 0;
        const pct = dur ? (cur / dur) * 100 : 0;
        range.value = Math.round(pct * 10);
        setRangeVisual(pct);
        setIcon();
    };

    const togglePlay = () => {
        if (video.paused || video.ended) video.play();
        else video.pause();
    };

    toggle.addEventListener('click', togglePlay);
    toggle.addEventListener('keydown', (e) => {
        if (e.key === 'Enter' || e.key === ' ') { e.preventDefault(); togglePlay(); }
    });

    video.addEventListener('click', togglePlay);

    let seeking = false;
    range.addEventListener('input', () => {
        seeking = true;
        const pct = (range.value / 1000) * 100;
        setRangeVisual(pct);
    });

    range.addEventListener('change', () => {
        const dur = video.duration || 0;
        if (dur) video.currentTime = (range.value / 1000) * dur;
        seeking = false;
    });

    video.addEventListener('loadedmetadata', syncUI);
    video.addEventListener('timeupdate', () => { if (!seeking) syncUI(); });
    video.addEventListener('play', setIcon);
    video.addEventListener('pause', setIcon);
    video.addEventListener('ended', setIcon);

    syncUI();

    modal.dataset.inited = '1';
};

document.querySelectorAll('.trigger').forEach(item => {
    item.addEventListener('click', () => {
        const targetId = item.id.replace('item-','');
        const target = document.querySelector(`.modal#modal-${targetId}`);
        if(!target) return;

        document.querySelectorAll('.modal').forEach(modal => modal.classList.remove('is-active'));

        initModal(target);

        const v = target.querySelector('video');
        if (v) {
            try { v.currentTime = 0; } catch (e) {}
            v.play();
            const r = target.querySelector('input[type="range"]');
            if (r) { r.value = 0; r.style.backgroundSize = '0% 100%'; }
            const ic = target.querySelector('.icon');
            if (ic) ic.className = 'icon play';
        }

        target.classList.add('is-active');
        
        updateModalNavButtons(target, getAllModals());
    });
});

document.querySelectorAll('.modal-close').forEach(item => {
    item.addEventListener('click', () => {
        document.querySelectorAll('.modal').forEach(modal => modal.classList.remove('is-active'));
        document.querySelectorAll('.modal video').forEach(v => v.pause());
    });
});

const scroller = document.getElementById("carousel");
if (scroller) {

    const btnLeft = document.getElementById("nav-button-left") || document.querySelector('.nav-button-left');
    const btnRight = document.getElementById("nav-button-right") || document.querySelector('.nav-button-right');

    if (btnLeft) btnLeft.addEventListener("click", () => snapBy(-1));
    if (btnRight) btnRight.addEventListener("click", () => snapBy(1));

    function getItems() {
        return Array.from(scroller.querySelectorAll(".carousel-site"));
    }

    function updateNavButtons(currentIndex, maxIndex) {
        if (!btnLeft || !btnRight) return;
        btnLeft.style.opacity = currentIndex > 0 ? "1" : "0";
        btnRight.style.opacity = currentIndex < maxIndex ? "1" : "0";
        btnLeft.style.pointerEvents = currentIndex > 0 ? "" : "none";
        btnRight.style.pointerEvents = currentIndex < maxIndex ? "" : "none";
    }

    function getCurrentIndex() {
        const items = getItems();
        if (!items.length) return 0;
        const left = scroller.scrollLeft;
        const center = left + scroller.clientWidth / 2;
        let closest = 0;
        let minDist = Infinity;
        for (let i = 0; i < items.length; i++) {
            const item = items[i];
            const itemCenter = item.offsetLeft + item.offsetWidth / 2;
            const d = Math.abs(itemCenter - center);
            if (d < minDist) {
                minDist = d;
                closest = i;
            }
        }
        return closest;
    }

    function snapBy(direction) {
        const items = getItems();
        if (!items.length) return;
        const currentIndex = getCurrentIndex();
        const nextIndex = Math.max(0, Math.min(items.length - 1, currentIndex + direction));
        scroller.scrollTo({ left: items[nextIndex].offsetLeft, behavior: "smooth" });
    }

    scroller.addEventListener("scroll", () => {
        const idx = getCurrentIndex();
        updateNavButtons(idx, getItems().length - 1);
    });

    window.addEventListener("resize", () => {
        const idx = getCurrentIndex();
        updateNavButtons(idx, getItems().length - 1);
    });

    updateNavButtons(getCurrentIndex(), getItems().length - 1);
}

function getAllModals() {
    return Array.from(document.querySelectorAll('.modal'));
}

function updateModalNavButtons(currentModal, allModals) {
    if (!currentModal) return;
    
    const currentIndex = allModals.indexOf(currentModal);
    const leftBtn = currentModal.querySelector('.modal-nav-left');
    const rightBtn = currentModal.querySelector('.modal-nav-right');
    
    if (leftBtn) {
        leftBtn.style.display = currentIndex > 0 ? 'block' : 'none';
    }
    if (rightBtn) {
        rightBtn.style.display = currentIndex < allModals.length - 1 ? 'block' : 'none';
    }
}

function switchModal(direction) {
    const currentModal = document.querySelector('.modal.is-active');
    if (!currentModal) return;

    const allModals = getAllModals();
    const currentIndex = allModals.indexOf(currentModal);
    
    if (currentIndex === -1) return;

    const newIndex = currentIndex + direction;
    
    if (newIndex < 0 || newIndex >= allModals.length) return;
    
    const targetModal = allModals[newIndex];
    
    if (!targetModal) return;

    const currentVideo = currentModal.querySelector('video');
    if (currentVideo) currentVideo.pause();
    currentModal.classList.remove('is-active');

    initModal(targetModal);
    
    const targetVideo = targetModal.querySelector('video');
    if (targetVideo) {
        try { targetVideo.currentTime = 0; } catch (e) {}
        targetVideo.pause();
        const r = targetModal.querySelector('input[type="range"]');
        if (r) { r.value = 0; r.style.backgroundSize = '0% 100%'; }
        const ic = targetModal.querySelector('.icon');
        if (ic) ic.className = 'icon play';
    }

    targetModal.classList.add('is-active');
    
    updateModalNavButtons(targetModal, allModals);
}

document.querySelectorAll('.modal-nav-left').forEach(button => {
    button.addEventListener('click', (e) => {
        e.stopPropagation();
        switchModal(-1);
    });
});

document.querySelectorAll('.modal-nav-right').forEach(button => {
    button.addEventListener('click', (e) => {
        e.stopPropagation();
        switchModal(1);
    });
});

function showPage(pageName) {
    document.querySelectorAll('.page-container').forEach(page => {
        page.classList.remove('active');
    });
    
    if (pageName === 'start') {
        document.getElementById('start-page').classList.add('active');
    } else if (pageName === 'main') {
        document.getElementById('main-page').classList.add('active');
    }
}